﻿// (c) Copyright 2025 Micro Focus or one of its affiliates.
// The only warranties for products and services of Micro Focus and its affiliates and licensors ("Micro Focus") are as may be set forth in the express warranty statements accompanying such products and services. Nothing herein should be construed as constituting an additional warranty. Micro Focus shall not be liable for technical or editorial errors or omissions contained herein. The information contained herein is subject to change without notice.

(function () {

const languageToDisplayName = new Map([
    ["python", "Python"],
    ["csharp", "C# .NET"],
    ["cpp", "C++"],
    ["c", "C"],
    ["java", "Java"]
])


function showHide(event)
{
    // Shows the code sample for the selected language and identifies the selected tab

    const target_lang = event.target.getAttribute('aria-controls')
    const target_container = getParentChooser(event.target)

    const tabs = target_container.querySelectorAll('div.codeSample')
    tabs.forEach(tab => {
        if (tab.id === target_lang)
        {
            tab.classList.add('active')
        }
        else
        {
            tab.classList.remove('active')
        }
    });

    const links = target_container.querySelectorAll('div.langSelect button')
    links.forEach(link => {
        link.setAttribute("aria-selected", false)
    });
    event.target.setAttribute("aria-selected", true)
}


function getParentChooser(link)
{
    // Given an element such as a tab button, returns the parent div.idolProgrammingLanguageChooser

    let element = link.parentElement;
    while (element !== null)
    {
        if ((element.tagName == "DIV") && (element.classList.contains("idolProgrammingLanguageChooser")))
        {
            return element;
        }

        element = element.parentElement;
    }
    console.error("Unable to locate parent div.idolProgrammingLanguageChooser")
    return null;
}


function getDefaultLanguage()
{
    // Reads the default programming language from a <meta> element in the HTML

    let result = null
    const meta_elem = document.querySelector("meta[name='idolProgrammingLanguageChooser-default'")
    if (meta_elem !== null && meta_elem.getAttribute('content') !== null)
    {
        result = meta_elem.getAttribute('content')
    }

    return result
}


function initializeLangChooser(element, chooser_id)
{
    const buttonGroup = document.createElement("div")
    buttonGroup.classList.add("langSelect")
    buttonGroup.setAttribute("role", "tablist")

    const tabs = element.querySelectorAll("div.codeSample");
    for (const [index, tab] of tabs.entries())
    {
        const lang = tab.getAttribute("data-idolProgrammingLanguageChooser-language")

        if (lang === null || languageToDisplayName.get(lang) === undefined)
        {
            continue
        }

        const tab_id = "idolProgrammingLanguageChooser" + chooser_id + "-Tab" + index
        const button_id = "idolProgrammingLanguageChooser" + chooser_id + "-Button" + index
        
        tab.id = tab_id
        tab.setAttribute("role", "tabpanel")
        tab.setAttribute("aria-labelledby", button_id)

        const button = document.createElement("button")
        button.id = button_id
        button.setAttribute("role", "tab")
        button.setAttribute("aria-controls", tab_id)
        button.appendChild( document.createTextNode(languageToDisplayName.get(lang)) )
        button.addEventListener('click', showHide, false)

        if (lang === getDefaultLanguage())
        {
            tab.classList.add("active")
            button.setAttribute("aria-selected", true)
        }
        
        buttonGroup.appendChild(button)
    }

    element.prepend(buttonGroup)
}


function allowChoosing()
{
    const langChoosers = document.querySelectorAll('div.idolProgrammingLanguageChooser');
    for (let i = 0; i < langChoosers.length; i++)
    {
        initializeLangChooser(langChoosers[i], i)
    }
}


if (document.readyState === 'loading')
{
    document.addEventListener('DOMContentLoaded', allowChoosing);
}
else
{
    allowChoosing();
}

})();